/**
* \file: main.c
*
* Program to test library and the error memory daemon.
* 
* This file implements tests for the error memory access library
* and the errror memory daemon.
*
* \component: errmem_lib
*
* \author: Markus Kretschmann (mkretschmann@de.adit-jv.com)
*
* \copyright (c) 2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
* <history item>
*/


#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <unistd.h>
#include <linux/errmem.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include "errmem_lib.h"

#define NUM_DEVICES           5

#define MAGIC_HEADER 0x22d9f147 // random number
typedef struct  __attribute__ ((packed)) ErrmemBackendBlockDeviceHeader {
	uint32_t magic;           //header magic
	uint16_t crc;             //header crc
	uint16_t header_length;   //total length of the current header
	uint32_t blocksize;       //blocksize with which the storage is created
	uint32_t persistent;      //number of persistent (non-overwritable) blocks
	uint32_t blocks;          //total number of blocks available in the storage
	uint32_t blockmagic;      //keeps the valid block magic for the current persistent storage
	uint32_t w_width;         //alignment with which this storage was created
} ErrmemBackendBlockDeviceHeader_t;

static uint16_t const lk_up_tbl_16[256] = {
	0x0000, 0xC0C1, 0xC181, 0x0140, 0xC301, 0x03C0, 0x0280, 0xC241, 
	0xC601, 0x06C0, 0x0780, 0xC741, 0x0500, 0xC5C1, 0xC481, 0x0440,
	0xCC01, 0x0CC0, 0x0D80, 0xCD41, 0x0F00, 0xCFC1, 0xCE81, 0x0E40,
	0x0A00, 0xCAC1, 0xCB81, 0x0B40, 0xC901, 0x09C0, 0x0880, 0xC841,
	0xD801, 0x18C0, 0x1980, 0xD941, 0x1B00, 0xDBC1, 0xDA81, 0x1A40,
	0x1E00, 0xDEC1, 0xDF81, 0x1F40, 0xDD01, 0x1DC0, 0x1C80, 0xDC41,
	0x1400, 0xD4C1, 0xD581, 0x1540, 0xD701, 0x17C0, 0x1680, 0xD641,
	0xD201, 0x12C0, 0x1380, 0xD341, 0x1100, 0xD1C1, 0xD081, 0x1040,
	0xF001, 0x30C0, 0x3180, 0xF141, 0x3300, 0xF3C1, 0xF281, 0x3240,
	0x3600, 0xF6C1, 0xF781, 0x3740, 0xF501, 0x35C0, 0x3480, 0xF441,
	0x3C00, 0xFCC1, 0xFD81, 0x3D40, 0xFF01, 0x3FC0, 0x3E80, 0xFE41,
	0xFA01, 0x3AC0, 0x3B80, 0xFB41, 0x3900, 0xF9C1, 0xF881, 0x3840,
	0x2800, 0xE8C1, 0xE981, 0x2940, 0xEB01, 0x2BC0, 0x2A80, 0xEA41,
	0xEE01, 0x2EC0, 0x2F80, 0xEF41, 0x2D00, 0xEDC1, 0xEC81, 0x2C40,
	0xE401, 0x24C0, 0x2580, 0xE541, 0x2700, 0xE7C1, 0xE681, 0x2640,
	0x2200, 0xE2C1, 0xE381, 0x2340, 0xE101, 0x21C0, 0x2080, 0xE041,
	0xA001, 0x60C0, 0x6180, 0xA141, 0x6300, 0xA3C1, 0xA281, 0x6240,
	0x6600, 0xA6C1, 0xA781, 0x6740, 0xA501, 0x65C0, 0x6480, 0xA441,
	0x6C00, 0xACC1, 0xAD81, 0x6D40, 0xAF01, 0x6FC0, 0x6E80, 0xAE41,
	0xAA01, 0x6AC0, 0x6B80, 0xAB41, 0x6900, 0xA9C1, 0xA881, 0x6840,
	0x7800, 0xB8C1, 0xB981, 0x7940, 0xBB01, 0x7BC0, 0x7A80, 0xBA41,
	0xBE01, 0x7EC0, 0x7F80, 0xBF41, 0x7D00, 0xBDC1, 0xBC81, 0x7C40,
	0xB401, 0x74C0, 0x7580, 0xB541, 0x7700, 0xB7C1, 0xB681, 0x7640,
	0x7200, 0xB2C1, 0xB381, 0x7340, 0xB101, 0x71C0, 0x7080, 0xB041,
	0x5000, 0x90C1, 0x9181, 0x5140, 0x9301, 0x53C0, 0x5280, 0x9241,
	0x9601, 0x56C0, 0x5780, 0x9741, 0x5500, 0x95C1, 0x9481, 0x5440,
	0x9C01, 0x5CC0, 0x5D80, 0x9D41, 0x5F00, 0x9FC1, 0x9E81, 0x5E40,
	0x5A00, 0x9AC1, 0x9B81, 0x5B40, 0x9901, 0x59C0, 0x5880, 0x9841,
	0x8801, 0x48C0, 0x4980, 0x8941, 0x4B00, 0x8BC1, 0x8A81, 0x4A40,
	0x4E00, 0x8EC1, 0x8F81, 0x4F40, 0x8D01, 0x4DC0, 0x4C80, 0x8C41,
	0x4400, 0x84C1, 0x8581, 0x4540, 0x8701, 0x47C0, 0x4680, 0x8641,
	0x8201, 0x42C0, 0x4380, 0x8341, 0x4100, 0x81C1, 0x8081, 0x4040
};

#define LOOKUP_VALUE(a, b) (lk_up_tbl_16[((a) ^ (b)) & 0xFF])
#define CALC_CRC16(a, b) ( ((a)>>8) ^ (LOOKUP_VALUE(a, b) & 0xFFFF) )

// calculates the crc16 checksum
static inline uint16_t calc_crc16(uint16_t seed, const uint8_t *verify , size_t len)
{
	uint16_t crc = seed;
	while (len--)
		crc = (CALC_CRC16(crc, *verify++) & 0xFFFF);
	return crc;
}

static int32_t execute_cmd(char* cmd)
{
    int32_t err = 0;

	if (cmd) {
		fprintf(stdout, "execute cmd %s\n", cmd);
		err = system(cmd);
		if (err < 0)
			fprintf(stdout, "execution of command failed: %s\n", cmd);
	}
	return err;
}

static int32_t rm_cmd(char* name)
{
    int32_t err = 0;

	/* command is: 
	 * mv filename filename.bck or cp filename filename.bck
	 */
	int32_t sz = 3 /* "rm " */+ strlen(name) +1;
	char *cmd = calloc(1, sz);
	if (cmd) {
		if (snprintf(cmd, sz,"rm %s", name) >= sz) {
			fprintf(stdout,"not enough memory allocated for command to remove file %s\n", name);
			err = -1;
		}
		fprintf(stdout, "execute cmd %s\n", cmd);
		err = system(cmd);
		if (err < 0)
			fprintf(stdout, "execution of command failed: %s\n", cmd);
		free(cmd);
	} else {
		fprintf(stdout, "can't get memory for command to remove file %s\n", name);
		err = -1;    	
	}
	return err;
}

static int32_t open_dev(char* name, int access_mode)
{
	int32_t err = 0;
	if (access_mode & O_RDWR || access_mode & O_WRONLY)
		access_mode |= O_SYNC;

	if (name) {
		err = open(name, access_mode);
		if (!~err) {
			err = -errno;
			fprintf(stdout, "cannot not open device: %s\n", name);
		}
	} else
		err = -EINVAL;
	return err;
}

static int32_t create_dev(char* name, uint blocksize, uint blocks)
{
    int32_t err = 0;

	/* command is: 
	 * /bin/dd if=/dev/zero of=/etc/filename bs=blocksize count=number
	 */
	int32_t sz = strlen(name) + strlen("/bin/dd if=/dev/zero of=") + 
		4 /*" bs="*/+ sizeof(blocksize) + 7 /*" count="*/+ 
		sizeof(blocks) + 1;

	char *cmd  = calloc(1, sz);

	if (cmd){
		if (snprintf(cmd, sz,"/bin/dd if=/dev/zero of=%s bs=%d count=%d",
					 name, blocksize, blocks) >= sz) {
			fprintf(stdout, "not enough memory allocated for command"
					" to create file %s\n", name);
			err = -1;
		} else {
			fprintf(stdout, "execute cmd %s\n", cmd);
			err = system(cmd);
			if (!~err) {
				err = -errno;
				fprintf(stdout, "execution of command failed: %s\n", cmd);
			} else {
				err = open_dev(name, O_RDWR);
				if (!~err) {
					err = -errno;
					fprintf(stdout, "can't open new created storage %s\n", name);
				}
			}
		}
		free(cmd);
	} else {
		fprintf(stdout, "error opening backend rc = %d\n", err);
		err = -1;
	}
    return err;
}

static int32_t write_dev(int32_t fd, char* name, char* buf, size_t size)
{
	int32_t err = 0;
	if (name && buf && size > 0 && fd >= 0) {
		err = write(fd, buf, size);
		if (!~err) {
			err = -errno;
			fprintf(stdout,"write failed on device %s\n", name);
		} else if (err >= 0 && (uint32_t)err < size) {
			fprintf(stdout, "write failed on device %s no space left - "
					"written bytes = %d bytes to write = %zd\n", name, err, size);
			err = -ENOSPC;
		} else
			err = 0;
	} else
		err = -EIO;
	return err;
}

int main (int argc, char* argv[])
{
	int rc     =  0;
	int fd     =  0;
	int pid_d  = -1;
	int nr_prc =  NUM_DEVICES - 1;
	int i      =  0;
//	struct errmem_message msg;

	char* fill  = "cat /proc/kallsyms > /dev/errmem";
	char* dev_1 = "/etc/errmem-storage_default.bin";
	char* dev_2 = "/etc/errmem-storage_2.bin";
	char* dev_3 = "/etc/errmem-storage_3.bin";
	char* dev_4 = "/etc/errmem-storage_4.bin";
	char* dev_5 = NULL;

	char* dev[NUM_DEVICES] = {"/etc/errmem-storage_default.bin",
							  "/etc/errmem-storage_2.bin",
							  "/etc/errmem-storage_3.bin",
							  "/etc/errmem-storage_4.bin",
							  NULL};

	char* logs[NUM_DEVICES] = {"/etc/errmem-storage_default.log",
							   "/etc/errmem-storage_2.log",
							   "/etc/errmem-storage_3.log",
							   "/etc/errmem-storage_4.log",
							  NULL};

	int blsz[NUM_DEVICES]  = {4096, 8192, 16384, 131072, 16384};
	int bkls[NUM_DEVICES]  = {1000, 200, 100, 10, 100};
	int pers[NUM_DEVICES]  = {10, 0, 2, 1, 3};
	int align[NUM_DEVICES] = { 4, 16, 32, 64, 64};

	if (argc < 2)
		fprintf(stdout, "usage /bin/errmem_test <name_of_raw_block_device>\n");
	else if (argc == 2) {
		dev_5 = argv[1];
		dev[NUM_DEVICES-1]= argv[1];
		logs[NUM_DEVICES-1] = "/etc/errmem-storage_5.log";
	} else
		fprintf(stdout, "too many parameter - usage /bin/errmem_test <name_of_raw_block_device>\n");

	fd = open_dev(dev_1, O_RDWR);
	if (~fd) {
		(void)close(fd);
		rm_cmd(dev_1);
		rm_cmd(logs[0]);
	} 
	fd = open_dev(dev_2, O_RDWR);
	if (~fd) {
		(void)close(fd);
		rm_cmd(dev_2);
		rm_cmd(logs[1]);
	}
	fd = open_dev(dev_3, O_RDWR);
	if (~fd) {
		(void)close(fd);
		rm_cmd(dev_3);
		rm_cmd(logs[2]);
	}
	fd = open_dev(dev_4, O_RDWR);
	if (~fd) {
		(void)close(fd);
		rm_cmd(dev_4);
		rm_cmd(logs[3]);
	}
	if (dev_5) {
		fd = open_dev(dev_5, O_RDWR);
		if (!~fd) {
			fprintf(stdout, "raw block device %s is not accessible\n", dev_5);
		} else
			rm_cmd(logs[4]);
	}

	/* create dev 1 to be to small */
	rc = create_dev(dev_1, blsz[0], bkls[1]);
	if (rc < 0)
		fprintf(stdout, "error creating device %s\n", dev_1);
	/* create dev 2 to be to big */
	rc = create_dev(dev_2, blsz[1], bkls[0]);
	if (rc < 0)
		fprintf(stdout, "error creating device %s\n", dev_1);
	/* correct system view */
	rc = create_dev(dev_3, blsz[2], bkls[2]);
	if (rc < 0)
		fprintf(stdout, "error creating device %s\n", dev_1);
	/* correct system view */
	rc = create_dev(dev_4, blsz[3], bkls[3]);
	if (rc < 0)
		fprintf(stdout, "error creating device %s\n", dev_1);

	fd = open_dev(dev_1, O_RDWR);
	if (0 <= fd) {
		ErrmemBackendBlockDeviceHeader_t hd = {0};
		hd.magic         = MAGIC_HEADER;
		hd.crc           = 0;
		hd.header_length = sizeof(hd);
		hd.blocksize     = blsz[0];
		hd.persistent    = pers[0];
		hd.blocks        = bkls[0];
		hd.blockmagic    = 0x12345678;
		hd.w_width       = align[0];
		hd.crc = calc_crc16(0,  (unsigned char*)&hd, sizeof(ErrmemBackendBlockDeviceHeader_t));
		rc = write_dev(fd, dev_1, (char*)&hd, sizeof(ErrmemBackendBlockDeviceHeader_t));
		if (rc < 0) {
			fprintf(stdout, "failed to write header information to device %s\n", dev_1);
		}
		close(fd);
	}

	fd = open_dev(dev_2, O_RDWR);
	if (0 <= fd) {
		ErrmemBackendBlockDeviceHeader_t hd = {0};
		hd.magic = MAGIC_HEADER;
		hd.crc   = 0;
		hd.header_length = sizeof(hd);
		hd.blocksize = blsz[1];
		hd.persistent = pers[1];
		hd.blocks    = bkls[1];
		hd.blockmagic    = 0x12345679;
		hd.w_width   = align[1];
		hd.crc = calc_crc16(0,  (unsigned char*)&hd, sizeof(ErrmemBackendBlockDeviceHeader_t));
		rc = write_dev(fd, dev_2, (char*)&hd, sizeof(ErrmemBackendBlockDeviceHeader_t));
		if (rc < 0) {
			fprintf(stdout, "failed to write header information to device %s\n", dev_2);
		}
		close(fd);
	}

	/* initialize the header with wrong header size */
	fd = open_dev(dev_3, O_RDWR);
	if (0 <= fd) {
		ErrmemBackendBlockDeviceHeader_t hd = {0};
		hd.magic = MAGIC_HEADER;
		hd.crc   = 0;
		hd.header_length = 0x18;
		hd.blocksize = blsz[2];
		hd.persistent = pers[2];
		hd.blocks    = bkls[2];
		hd.w_width   = align[2];
		hd.blockmagic    = 0x12345678;
		hd.crc = calc_crc16(0,  (unsigned char*)&hd, sizeof(ErrmemBackendBlockDeviceHeader_t));
		rc = write_dev(fd, dev_3, (char*)&hd, sizeof(ErrmemBackendBlockDeviceHeader_t));
		if (rc < 0) {
			fprintf(stdout, "failed to write header information to device %s\n", dev_3);
		}
		close(fd);
	}

	fd = open_dev(dev_4, O_RDWR);
	if (0 <= fd) {
		ErrmemBackendBlockDeviceHeader_t hd = {0};
		hd.magic = MAGIC_HEADER;
		hd.crc   = 0;
		hd.header_length = sizeof(hd);
		hd.blocksize = blsz[3];
		hd.persistent = pers[3];
		hd.blocks    = bkls[3] + 1;
		hd.w_width   = align[3];
		hd.blockmagic    = 0x12345678;
		hd.crc = calc_crc16(0,  (unsigned char*)&hd, sizeof(ErrmemBackendBlockDeviceHeader_t));
		rc = write_dev(fd, dev_4, (char*)&hd, sizeof(ErrmemBackendBlockDeviceHeader_t));
		if (rc < 0) {
			fprintf(stdout, "failed to write header information to device %s\n", dev_4);
		}
		close(fd);
	}

	fd = open_dev(dev_5, O_RDWR);
	if (0 <= fd) {
		ErrmemBackendBlockDeviceHeader_t hd = {0};
		hd.magic = 0x28d9f147;
		hd.crc   = 0;
		hd.header_length = 0x18;
		hd.blocksize = blsz[4];
		hd.persistent = pers[4];
		hd.blocks    = bkls[4];
		hd.w_width   = align[4];
		hd.blockmagic    = 0x12345678;
		hd.crc = calc_crc16(0,  (unsigned char*)&hd, sizeof(ErrmemBackendBlockDeviceHeader_t));
		rc = write_dev(fd, dev_5, (char*)&hd, sizeof(ErrmemBackendBlockDeviceHeader_t));
		if (rc < 0) {
			fprintf(stdout, "failed to write header information to device %s\n", dev_5);
		}
		close(fd);
	}
	
	/* start the daemon from here */
	int32_t sz = strlen("/bin/errmemd -d -s ") + strlen(dev_1) + 
		6 /*" --bs="*/+ sizeof(int) + 7 /*" --cnt="*/+ 
		sizeof(int) + 4 /* -e */ + sizeof(int) +5 /* -o1 */ + 4/* -w */ +
		sizeof(int);
	sz += strlen(dev_2) + 6 /*" --bs="*/+ sizeof(int) + 7 /*" --cnt="*/+ 
		sizeof(int) + 4 /* -e */ + sizeof(int) +5 /* -o1 */ + 4/* -w */ +
		sizeof(int);
	sz += strlen(dev_3) + 6 /*" --bs="*/+ sizeof(int) + 7 /*" --cnt="*/+ 
		sizeof(int) + 4 /* -e */ + sizeof(int) +5 /* -o1 */ + 4/* -w */ +
		sizeof(int);
	sz += strlen(dev_4) + 6 /*" --bs="*/+ sizeof(int) + 7 /*" --cnt="*/+ 
		sizeof(int) + 4 /* -e */ + sizeof(int) +5 /* -o1 */ + 4/* -w */ +
		sizeof(int);
	if (dev_5)
		sz += strlen(dev_5) + 6 /*" --bs="*/+ sizeof(int) + 7 /*" --cnt="*/+ 
			sizeof(int) + 4 /* -e */ + sizeof(int) +5 /* -o */ + 4/* -w */ +
			sizeof(int);
    sz += strlen(" &");
	sz += 1;

	char *cmd  = calloc(1, sz);

	if (0 <= rc && cmd){
		if (dev_5) {
			if (snprintf(cmd, sz,"/bin/errmemd -d -s %s --bs=%d --cnt=%d -e %d -o1 -w %d "
						 "-s %s --bs=%d --cnt=%d -e %d -o1 -w %d -s %s --bs=%d --cnt=%d -e %d -o1 -w %d "
						 "-s %s --bs=%d --cnt=%d -e %d -o1 -w %d -s %s --bs=%d --cnt=%d -e %d -o -w %d%s",
						 dev_1, blsz[0], bkls[0], pers[0], align[0],
						 dev_2, blsz[1], bkls[1], pers[1], align[1],
						 dev_3, blsz[2], bkls[2], pers[2], align[2],
						 dev_4, blsz[3], bkls[3], pers[3], align[3],
						 dev_5, blsz[4], bkls[4], pers[4], align[4]," &") >= sz) {
				fprintf(stdout, "not enough memory allocated for command to start daemon\n");
				rc = -1;
			}
		} else {
			if (snprintf(cmd, sz,"/bin/errmemd -d -s %s --bs=%d --cnt=%d -e %d -o1 -w %d "
						 "-s %s --bs=%d --cnt=%d -e %d -o1 -w %d -s %s --bs=%d --cnt=%d -e %d -o1 -w %d "
						 "-s %s --bs=%d --cnt=%d -e %d -o1 -w %d%s",
						 dev_1, blsz[0], bkls[0], pers[0], align[0],
						 dev_2, blsz[1], bkls[1], pers[1], align[1],
						 dev_3, blsz[2], bkls[2], pers[2], align[2],
						 dev_4, blsz[3], bkls[3], pers[3], align[3]," &") >= sz) {
				fprintf(stdout, "not enough memory allocated for command to start daemon\n");
				rc = -1;		
			}
		}
		if (0 <= rc) {
			pid_d = execute_cmd(cmd);
			fprintf(stdout, "deamon started - value pid_d = %d\n", pid_d);
			if (pid_d < 0)
				rc = pid_d;
		}
		free(cmd);
		cmd = NULL;
	} else {
		fprintf(stdout, "not enough memory allocated for command to start daemon\n");
		rc = -ENOMEM;
	}

	if (0 <= rc)
		rc = execute_cmd(fill);

	if (0 <= rc) {
		if (dev_5)
			nr_prc += 1;

		for (i = 1; i <= nr_prc; i++) {
			char* logfile  = logs[i-1];
			pid_t pID = fork();
			if (pID == 0) {
				int cfd    = -1;
				int sh     = -1;
				char* cdev = dev[i-1];
				struct errmem_message msg;
				int rcc    = 0;
				/* child code here */
				fprintf(stdout, "Log output for device %d in %s\n", i, logfile);
				cfd = open_dev(logfile, O_RDWR | O_CREAT);
				if (cfd >= 0) {
					fprintf(stdout, "successfully opened logfile %s\n", logfile);
					dprintf(cfd, "successfully opened logfile %s\n", logfile);
					sh = errmem_backend_open_session();
					if (sh <= 0) {
						dprintf(cfd, "errmem_backend_open_session (backend %d) returns error %d\n", i, sh);
					} else {
						sleep(10);
						dprintf(cfd, "backend opened sh = %d\n", sh);
						if (i != 1){
							rcc = errmem_backend_set_storage(sh, i);
							if (rcc < 0)
								dprintf(cfd, "cannot assign storage %s to session %d\n", cdev, i);
							else
								dprintf(cfd, "successfully assigned storage %s to session %d\n", cdev, i);
						}
						while (rcc == 0) {
							rcc = errmem_backend_read(sh, &msg);
							if (rcc < 0)
								dprintf(cfd, "error reading backend %s: error code = %d\n", cdev, rcc);
							else if (rcc == 0) {
								if(msg.length > ERRMEM_MAX_ENTRY_LENGTH) 
									msg.length = ERRMEM_MAX_ENTRY_LENGTH;
								if(msg.message[msg.length-1])
									msg.message[msg.length-1] = '\0';
								if(msg.message[msg.length-2] != '\n')
									msg.message[msg.length-2] = '\n';
								dprintf(cfd, "errmem_test: type = %d seqnum = %d clock = %d %d offset = %d flags = %d crc = %d storage = %d length = %d msg = %s\n", msg.type, msg.internal.seqnum, msg.internal.local_clock[0], msg.internal.local_clock[1], msg.internal.offset, msg.internal.flags, msg.internal.crc, msg.internal.storage, msg.length, msg.message);
							}
						}
						errmem_backend_close_session(sh);
						dprintf(cfd, "session with backend %s closed\n", cdev);
					}
					close(cfd);
				} else
					fprintf(stdout, "failed to open logfile %s\n", logfile);
				exit(0);
			} else if (pID < 0) {
				fprintf (stdout, "failed to fork child %d\n", i);
			} else {
				sleep(15);
				rc = execute_cmd(fill);
			}		
		}
	}
	if (cmd)
		free(cmd);
	return rc;
}

#if 0
	sh = errmem_backend_open_session();
	if (sh <= 0) {
		fprintf(stdout, "error opening backend rc = %d\n", rc);
		return 0;
	}
	else {
		fprintf(stdout, "backend opened sh = %d\n", sh);
	}

	while (rc == 0) {
		rc = errmem_backend_read(sh, &msg);
		if (rc < 0)
			fprintf(stdout, "error reading backend: error code = %d", rc);
		else
			fprintf(stdout, "errmem_test: type = %d seqnum = %d clock = %d %d offset = %d flags = %d crc = %d storage = %d length = %d msg = %s\n", msg.type, msg.internal.seqnum, msg.internal.local_clock[0], msg.internal.local_clock[1], msg.internal.offset, msg.internal.flags, msg.internal.crc, msg.internal.storage, msg.length, msg.message);
	}
	errmem_backend_close_session(sh);
	return 0;
}
#endif
#if 0
int main (int argc, char* argv[])
{
	int rc  = 0;
	int sh  = -1;
	struct errmem_message msg;
	argc = argc;
	argv = argv;
	sh   = sh;

	rc = errmem_backend_open_session();
	sh = rc;
	if (rc <= 0)
		fprintf(stdout, "error opening backend rc = %d\n", rc);
	else {
		fprintf(stdout, "backend opened sh = %d\n", sh);
		rc = 0;
	}

	if (sh > 0)
		fprintf(stdout, "Start Reading Default Device errmem-storage_140A085_150\n");	

	while (rc == 0) {
		rc = errmem_backend_read(sh, &msg);
		if (rc < 0)
			fprintf(stdout, "error reading backend errmem-storage_140A085_150 rc = %d\n", rc);
		else
			fprintf(stdout, "errmem_test: type = %d seqnum = %d clock = %d %d offset = %d flags = %d crc = %d storage = %d length = %d msg = %s\n", msg.type, msg.internal.seqnum, msg.internal.local_clock[0], msg.internal.local_clock[1], msg.internal.offset, msg.internal.flags, msg.internal.crc, msg.internal.storage, msg.length, msg.message);
	}

	rc = errmem_backend_set_storage(sh, 2);
	if (rc < 0)
		fprintf(stdout, "Cannot assign devic 2 - Errormemory_14.4A001 rc = %d\n", rc);
	else {
		fprintf(stdout, "Storage Errormemory_14.4A001 assigned\n");
		rc = 0;
	}

	if (rc == 0 && sh > 0)
		fprintf(stdout, "Start Reading Device Errormemory_14.4A001\n");	

	while (rc == 0) {
		rc = errmem_backend_read(sh, &msg);
		if (rc < 0)
			fprintf(stdout, "error reading backend Errormemory_14.4A001 rc = %d\n", rc);
		else
			fprintf(stdout, "errmem_test: type = %d seqnum = %d clock = %d %d offset = %d flags = %d crc = %d storage = %d length = %d msg = %s\n", msg.type, msg.internal.seqnum, msg.internal.local_clock[0], msg.internal.local_clock[1], msg.internal.offset, msg.internal.flags, msg.internal.crc, msg.internal.storage, msg.length, msg.message);
	}

	rc = errmem_backend_set_storage(sh, 3);
	if (rc < 0)
		fprintf(stdout, "Cannot assign devic 2 - Errormemory_14.4A002 rc = %d\n", rc);
	else {
		fprintf(stdout, "Storage Errormemory_14.4A002 assigned\n");
		rc = 0;
	}

	if (rc == 0 && sh > 0)
		fprintf(stdout, "Start Reading Device Errormemory_14.4A002\n");	

	while (rc == 0) {
		rc = errmem_backend_read(sh, &msg);
		if (rc < 0)
			fprintf(stdout, "error reading backend Errormemory_14.4A001 rc = %d\n", rc);
		else
			fprintf(stdout, "errmem_test: type = %d seqnum = %d clock = %d %d offset = %d flags = %d crc = %d storage = %d length = %d msg = %s\n", msg.type, msg.internal.seqnum, msg.internal.local_clock[0], msg.internal.local_clock[1], msg.internal.offset, msg.internal.flags, msg.internal.crc, msg.internal.storage, msg.length, msg.message);
	}

	return rc;
}

int main (int argc, char* argv[])
{
	int rc  = 0;
	int sh  = -1;
	int i = 0, j = 0;
	int ia[1000];
//	struct errmem_message msg;
	argc = argc;
	argv = argv;
	sh   = sh;
	i = i;
	j = j;

	for (; i < 1000; i++) {
		rc = errmem_backend_open_session();
		ia[i] = rc;
		if (rc <= 0)
			fprintf(stdout, "error opening backend rc = %d\n", rc);
		else
			fprintf(stdout, "backend opened sh = %d\n", ia[i]);
	}

	for (; j < 1000; j++) {
		rc = errmem_backend_close_session(ia[j]);
		if (rc < 0)
			fprintf(stdout, "error closing backend rc = %d\n", rc);
		else
			fprintf(stdout, "backend closed sh = %d\n", ia[j]);
	}

	rc = errmem_backend_open_session();
	sh = rc;
	if (rc <= 0)
		fprintf(stdout, "error opening backend rc = %d\n", rc);
	else {
		fprintf(stdout, "backend opened sh = %d\n", sh);
		rc = 0;
	}
	while (rc == 0) {
		rc = errmem_backend_read(sh, &msg);
		if (rc < 0)
			fprintf(stdout, "error reading backend rc = %d\n", rc);
		else
			fprintf(stdout, "errmem_test: type = %d seqnum = %d clock = %d %d offset = %d flags = %d crc = %d storage = %d length = %d msg = %s\n", msg.type, msg.internal.seqnum, msg.internal.local_clock[0], msg.internal.local_clock[1], msg.internal.offset, msg.internal.flags, msg.internal.crc, msg.internal.storage, msg.length, msg.message);
	}

	if (sh > 0)
		rc = errmem_backend_close_session(sh);

	if (rc < 0)
		fprintf(stdout, "error closing backend rc = %d\n", rc);
	else
		fprintf(stdout, "backend closed sh = %d\n", sh);


	rc = errmem_backend_open_session();
	sh = rc;
	if (rc <= 0)
		fprintf(stdout, "error opening backend rc = %d\n", rc);
	else {
		fprintf(stdout, "backend opened sh = %d\n", sh);
		rc = 0;
	}

	rc = errmem_backend_erase(sh, 0);
	if (rc < 0)
		fprintf(stdout, "error erasing backend 1 rc = %d\n", rc);
	else {
		fprintf(stdout, "backend 1 erased sh = %d status = %d\n", sh, rc);
		rc = 0;
	}

	while (rc == 0) {
		int x = 0;
		rc = errmem_backend_read(sh, &msg);
		if (rc < 0)
			fprintf(stdout, "error reading backend rc = %d\n", rc);
		else {
			fprintf(stdout, "errmem_test: type = %d seqnum = %d clock = %d %d offset = %d flags = %d crc = %d storage = %d length = %d msg = %s\n", msg.type, msg.internal.seqnum, msg.internal.local_clock[0], msg.internal.local_clock[1], msg.internal.offset, msg.internal.flags, msg.internal.crc, msg.internal.storage, msg.length, msg.message);

			if (x == 100)
				rc = errmem_backend_stop(sh);
			x++;
		}
	}

	rc = errmem_backend_stop(sh);
	
	rc = errmem_backend_erase(sh, 2);
	if (rc < 0)
		fprintf(stdout, "error erasing backend 2 rc = %d\n", rc);
	else {
		fprintf(stdout, "backend 2 erased sh = %d status = %d\n", sh, rc);
		rc = 0;
	}

	if (sh > 0)
		rc = errmem_backend_close_session(sh);

	if (rc < 0)
		fprintf(stdout, "error closing backend rc = %d\n", rc);
	else
		fprintf(stdout, "backend closed sh = %d\n", sh);

	rc = errmem_backend_open_session();
	sh = rc;
	if (rc <= 0)
		fprintf(stdout, "error opening backend rc = %d\n", rc);
	else {
		fprintf(stdout, "backend opened sh = %d\n", sh);
		rc = 0;
	}

	if (!rc){
		rc = errmem_backend_set_storage(sh, 0);
		if (rc < 0)
			fprintf(stdout, "failed to remove assignement of storage rc = %d\n", rc);
		else {
			fprintf(stdout, "assignement of storage is done sh = %d\n", sh);
			rc = 0;
		}
		rc = errmem_backend_set_storage(sh, 6);
		if (rc < 0)
			fprintf(stdout, "failed to remove assignement of storage rc = %d\n", rc);
		else {
			fprintf(stdout, "assignement of storage is done sh = %d\n", sh);
			rc = 0;
		}
		rc = errmem_backend_set_storage(sh, -5);
		if (rc < 0)
			fprintf(stdout, "failed to remove assignement of storage rc = %d\n", rc);
		else {
			fprintf(stdout, "assignement of storage is done sh = %d\n", sh);
			rc = 0;
		}
		rc = errmem_backend_set_storage(sh, 1);
		if (rc < 0)
			fprintf(stdout, "failed to remove assignement of storage rc = %d\n", rc);
		else {
			fprintf(stdout, "assignement of storage is done sh = %d\n", sh);
			rc = 0;
		}

		rc = errmem_backend_stop(sh);

		while (rc == 0) {
			rc = errmem_backend_read(sh, &msg);
			if (rc < 0)
				fprintf(stdout, "error reading backend rc = %d\n", rc);
			else
				fprintf(stdout, "errmem_test: type = %d seqnum = %d clock = %d %d offset = %d flags = %d crc = %d storage = %d length = %d msg = %s\n", msg.type, msg.internal.seqnum, msg.internal.local_clock[0], msg.internal.local_clock[1], msg.internal.offset, msg.internal.flags, msg.internal.crc, msg.internal.storage, msg.length, msg.message);
		}
	}
	return rc;
}
#endif
		
